#!/usr/bin/env python3
\"\"\"
run_Goal_TDColors.py - Launch After Effects batch_Goal_TDColors.jsx

Usage:
    python3 run_Goal_TDColors.py LEAGUE CSV_PATH PROJECT_AEP OUT_DIR LOGO_DIR [ANIM_NAME]

Example:
    python3 run_Goal_TDColors.py NHL /path/to/teams_rgb.csv /path/to/Project.aep /path/to/output /path/to/Logos GOAL_1_3
\"\"\"
import os
import subprocess
import sys
from pathlib import Path

def main():
    if len(sys.argv) < 6:
        print(\"Usage: python3 run_Goal_TDColors.py LEAGUE CSV_PATH PROJECT_AEP OUT_DIR LOGO_DIR [ANIM_NAME]\")
        sys.exit(1)

    league = sys.argv[1]
    csv_path = Path(sys.argv[2]).expanduser().resolve()
    project_aep = Path(sys.argv[3]).expanduser().resolve()
    out_dir = Path(sys.argv[4]).expanduser().resolve()
    logo_dir = Path(sys.argv[5]).expanduser().resolve()
    anim_name = sys.argv[6] if len(sys.argv) > 6 else \"GOAL_1_3\"

    # Ensure files/dirs exist
    if not csv_path.exists():
        print(f\"CSV not found: {csv_path}\")
        sys.exit(2)
    if not project_aep.exists():
        print(f\"Project AEP not found: {project_aep}\")
        sys.exit(3)
    out_dir.mkdir(parents=True, exist_ok=True)
    if not logo_dir.exists():
        print(f\"Logo dir not found: {logo_dir}\")
        sys.exit(4)

    # Common JSX include
    jsx_common = Path(os.environ.get(\"AE_COMMON_JSX\", \"./gl_common.jsxinc\")).expanduser().resolve()
    if not jsx_common.exists():
        print(f\"AE_COMMON_JSX (gl_common.jsxinc) not found at: {jsx_common}\")
        sys.exit(5)

    # JSX must live next to this script
    jsx_batch = Path(__file__).parent / \"batch_Goal_TDColors.jsx\"
    if not jsx_batch.exists():
        print(f\"JSX file not found next to this script: {jsx_batch}\")
        sys.exit(6)

    # Layer defaults + pipeline envs
    envs = {
        \"AE_COMMON_JSX\": str(jsx_common),
        \"AE_PROJECT\": str(project_aep),
        \"AE_CSV\": str(csv_path),
        \"AE_COMP\": \"Goal-192\",   # CHANGE if your comp is different
        \"AE_LEAGUE\": league,
        \"AE_LOGO_DIR\": str(logo_dir),
        \"AE_LOGO_PATH_TEMPLATE\": \"{league}/{abbr}\",
        \"AE_LOGO_EXTS\": \"png,jpg,jpeg,svg,ai,psd\",
        \"AE_OUTDIR\": str(out_dir),
        \"AE_PATH_TEMPLATE\": \"{league}/{abbr}\",
        \"AE_ANIM\": anim_name,
        \"AE_RS_TEMPLATE\": \"Best Settings\",
        \"AE_OM_TEMPLATE\": \"PNG Sequence\",
        \"AE_PURGE_BEFORE_RENDER\": \"1\",
        \"AE_NO_RENDER\": \"0\",
        \"AE_QUIT\": \"1\",
        # Layer mapping
        \"AE_LAYER_GOAL_TEXT\": \"GOAL!\",
        \"AE_LAYER_TD_PRE\": \"TD PreComp\",
        \"AE_SOLID_LAYER\": \"Solid\",
        \"AE_TEAMNAME_LAYER\": \"TeamName\",
        \"AE_LAYER_LOGO_PRECOMP\": \"LogoPreComp\",
        \"AE_SUBLAYER_LOGO_IN_PRECOMP\": \"Logo\",
        \"AE_LAYER_LOGO\": \"Logo\",
    }

    # Merge with current env
    new_env = os.environ.copy()
    new_env.update(envs)

    # Launch After Effects (macOS) via AppleScript
    after_effects_app = os.environ.get(\"AE_APP_NAME\", \"Adobe After Effects 2025\")

    # If After Effects isn't running, this opens it and runs the JSX; if running, it just runs the JSX.
    cmd = [\"osascript\", \"-e\", f'tell application \"{after_effects_app}\" to DoScriptFile \"{jsx_batch}\"']

    print(\"Launching After Effects batch script…\")
    print(\"Command:\", \" \".join(cmd))
    proc = subprocess.run(cmd, env=new_env)

    if proc.returncode != 0:
        print(f\"After Effects returned error code {proc.returncode}\")
        sys.exit(proc.returncode)
    else:
        print(\"✅ Render complete.\")

if __name__ == \"__main__\":
    main()
